function OPTIONS = gcplot(X,labels,OPTIONS)

% GCPLOT  Genuine Clusters PLOT
%
% gcplot(X) displays data that are stored in the rows of the NxD matrix X,
% where D=1 corresponds to a column vector. For convenience,
% X is also allowed to be a row vector, i.e., 1-by-N.
%
% gcplot(X, labels) displays the different clusters comprising the data
% (in different colors and using different markers). The N-vector 'labels' is
% assumed to take values in the continuous integer set {1, 2, ..., K}.
% When labels is zero or negative, the corresponding points are
% outliers (will be represented by black dots).
%
% gcplot(X, labels, OPTIONS) allows you to specify the values of the
% following fields of OPTIONS:
%   .markerSize: default = 10;
%   .markers: an array of symbols used for displaying the clusters
%       (default: '.' for a single cluster; 'o+v*d' for first 5 clusters and
%        randomly chosen thereafter)
%   .colors: a K-by-3 matrix of RGB values (default colors given for first 5 clusters,
%       and randomly chosen thereafter)
%   .view: this is only useful when D>3
%       'first3': plot only X(:,1:3)
%       'last3': plot only X(:, D-2:D)
%       'rand3': plot X(:, randsample(D,3))
%       'pca' (default): plot the first three principal components of the data
%
% OPTIONS = gcplot(X, labels, OPTIONS) outputs all the options that were
% used in generating the plot.
%
% (c) 2010 Guangliang Chen, Duke University, glchen@math.duke.edu

%%
[N,D] = size(X);

if N == 1 % row vector
    X = X'; % convert to column vector
    N = D;
    D = 1;
end

if nargin<2 || isempty(labels)
    labels = ones(N,1); % one cluster
end

if nargin<3
    OPTIONS = struct();
end

if ~isfield(OPTIONS, 'markerSize')
    OPTIONS.markerSize = 10;
end

if ~isfield(OPTIONS, 'markers')
    if any(labels~=1) % The data has outliers or multiple clusters
        OPTIONS.markers ='o+v*d';
    else % no outliers and one single cluster
        OPTIONS.markers = '.';
    end
end

if ~isfield(OPTIONS,'colors')
    if any(labels~=1)
        OPTIONS.colors = [1 0 0; 0 0.4 0; 0 0 1; 1 0.2 0.6; 0.8 0.4 0];
    else
        OPTIONS.colors = [0,0,1];
    end
end

K = max(labels); % number of clusters
n_markers = length(OPTIONS.markers); % number of markers supplied

if K>n_markers % more clusters
    %OPTIONS.markers(n_markers+1:K) = randsample('o+v*dxsph^<>', K-n_markers, true);
    symbols = 'o+v*dxsph^<>';
    OPTIONS.markers(n_markers+1:K) = symbols(ceil(numel(symbols)*rand(1,K-n_markers)));
    OPTIONS.colors(n_markers+1:K, :) = rand(K-n_markers,3);
elseif K<n_markers % less or equal
    OPTIONS.markers =OPTIONS.markers(1:K);
    OPTIONS.colors =OPTIONS.colors(1:K,:);
end

%% large ambient dimension
if D > 3
    
    if ~isfield(OPTIONS, 'view')
        OPTIONS.view = 'pca';
    end
    
    switch lower(OPTIONS.view)
        case 'first3'
            X = X(:,1:3);
        case 'last3'
            X = X(:,D-2:D);
        case 'rand3'
            X = X(:, randsample(D,3));
        case 'pca'
            [U,S] = svds(X - repmat(mean(X,1),N,1), 6);
            X = U(:,1:3).*repmat(transpose(diag(S(1:3,1:3))), N,1);
    end
    
end

%%
inds_0 = find(labels<1); % outliers

%figure
%axes('position',[0  0  1  1])

hold on

if D>=3
    
    for k = 1:K
        inds_k = find(labels==k);
        plot3(X(inds_k,1), X(inds_k,2), X(inds_k,3),OPTIONS.markers(k), 'color',OPTIONS.colors(k,:), 'MarkerSize',OPTIONS.markerSize)
    end
    plot3(X(inds_0,1),X(inds_0,2),X(inds_0,3),'k.','MarkerSize',OPTIONS.markerSize)
    %axis equal
    
elseif D==2
    
    for k = 1:K
        inds_k = find(labels==k);
        plot(X(inds_k,1), X(inds_k,2),OPTIONS.markers(k), 'color',OPTIONS.colors(k,:),'MarkerSize',OPTIONS.markerSize)
    end
    plot(X(inds_0,1),X(inds_0,2),'k.','MarkerSize',OPTIONS.markerSize)
    %axis equal
    
else %% D=1
    
    for k = 1:K
        inds_k = find(labels==k);
        plot(inds_k, X(inds_k,1),OPTIONS.markers(k), 'color',OPTIONS.colors(k,:),'MarkerSize',OPTIONS.markerSize)
    end
    plot(inds_0, X(inds_0,1),'k.','MarkerSize',OPTIONS.markerSize)
    
end

hold off

axis tight

% % %set(gca, 'XTick',[],'YTick', [])
% if D >= 3
%     set(gca,'XTick',[], 'XTickLabel', {},'YTick', [], 'YTickLabel', [] ,'ZTick', [], 'ZTickLabel', {} )
% else
%     set(gca,'XTick',[], 'XTickLabel', {},'YTick', [], 'YTickLabel', {})
% end
% box on
