function labels = specluster(X, k, opts)

% Spectral clustering function
%
% Input
%   X: n by d data matrix
%   k: number of clusters
%   opts: a structure array with the following fields
%       .sigmaValue: radius or # nearest neighbors (default=7) depending on sigmaMethod below
%       .sigmaMethod: a string, one of the following
%                'sigma': in this case sigmaValue refers to the sigma in the gaussian kernel
%                'meanDist' (default): mean distance of points to their sigmaValue nearest neighbors
%                'medianDist': median distance of points to their sigmaValue nearest neighbors
%                'selfTuning': sigmaValue = # nearest neighbors
%       .normalizeL: whether or not to normalize the graph Laplacian. Default = true
%       .clusterMethod: one of 'Ncut' (default) or 'NJW'
% Ouput
%   labels: clusters labels found by Ncut

%%
if nargin<3
    opts = struct();
end

if ~isfield(opts, 'sigmaMethod')
    opts.sigmaMethod = 'meanDist';
end

if ~isfield(opts, 'sigmaValue')
    opts.sigmaValue = 7;
end

if ~isfield(opts, 'normalizeL')
    opts.normalizeL = true;
end

if ~isfield(opts, 'clusterMethod')
    opts.clusterMethod = 'Ncut';
end

%%
n = size(X,1); % number of data points

norms2 = sum(X.^2,2);
dists2 = repmat(norms2,1,n) + repmat(norms2',n,1) - (2*X)*X';
dists2_sort = sort(dists2, 2, 'ascend');

switch opts.sigmaMethod
    case 'sigma'
        sigma2 = 2*opts.sigmaValue^2;
    case 'meanDist'
        sigma2 = 2*mean(sqrt(dists2_sort(:, opts.sigmaValue)))^2;
    case 'medianDist'
        sigma2 = 2*median(sqrt(dists2_sort(:, opts.sigmaValue)))^2;
    case 'selfTuning'
        sigma2 = 2*sqrt(dists2_sort(:,opts.sigmaValue))*sqrt(dists2_sort(:,opts.sigmaValue))';
end

W = exp(-dists2 ./ sigma2);
W(1:n+1:end) = 0;
W = (W+W')/2;
figure; imagesc(W)

if opts.normalizeL
    dvec_inv = 1./sqrt(sum(W,2));
    
    W_tilde = repmat(dvec_inv, 1, n).*W.*repmat(dvec_inv', n, 1);
    W_tilde = (W_tilde+W_tilde')/2;
else
    W_tilde = W; % unnormalized spectral clustering
end

[V,~] = eigs(W_tilde, k, 'LM', struct('issym', true, 'isreal', true));

switch opts.clusterMethod
    case 'Ncut'
        V = repmat(dvec_inv, 1, k-1).*V(:,2:k);
    case 'NJW'
        V = V./repmat(sqrt(sum(V.^2,2)), 1,k);
end

labels = litekmeans(V', k);
%labels = kmeans(V, k, 'replicates', 10);
% for i = 1:k-1
%     figure; gcplot(V(:,i),labels)
% end

figure; gcplot(V,labels)
